import { Component, EventEmitter, OnInit, Output, ViewChild } from '@angular/core';
import { NgForm } from '@angular/forms';
import { Router } from '@angular/router';
import { ConfirmationService } from 'primeng/api';
import { takeUntil } from 'rxjs/operators';
import { UtilsService } from '../../../../oneit/services/utils.service';
import { BaseComponent } from '../../../base/base.component';
import { CAREER_TABS } from '../../../config/constants';
import { CareerProfileModel } from '../../../models/career-profile.model';
import { CareerSkillModel, SkillModel } from '../../../models/career-skill.model';
import { HelperService } from '../../../services/helper.service';
import { PersonalDetailsService } from '../../../services/personal-details.service';

@Component({
    selector: 'app-skills',
    templateUrl: './skills.component.html',
    styleUrls: ['./skills.component.scss']
})
export class SkillsComponent extends BaseComponent implements OnInit {

    assocs = ['Skills.Skill'];
    skills: SkillModel[] = [];
    careerProfile = new CareerProfileModel();
    ratings = [];

    @ViewChild('form') form: NgForm;
    @Output() skillsSaved = new EventEmitter();
    @Output() careerProfileLoaded = new EventEmitter<CareerProfileModel>();

    constructor(
        private personalDetailsService: PersonalDetailsService,
        private confirmationService: ConfirmationService,
        private utilsService: UtilsService,
        private helperService: HelperService,
        private router: Router
    ) {
        super();
    }

    ngOnInit() {
        this.getCareerProfile();
        this.getSkillRatings();
    }

    getCareerProfile(): void {
        this.isLoading = true;
        this.personalDetailsService.getCareerProfile(this.assocs)
            .pipe(takeUntil(this.componentInView))
            .subscribe(response => {
                this.isLoading = false;
                this.careerProfile = response;
                this.updatedObjs[this.careerProfile.ObjectID] = this.careerProfile;
                this.utilsService.addObjsToJSONByObjectID(this.updatedObjs, this.careerProfile.Skills);
                this.addSkill(true);
                this.careerProfileLoaded.emit(this.careerProfile);
            }, err => {
                this.isLoading = false;
                this.utilsService.handleError(err);
            })
    }

    getSkills($event): void {
        this.personalDetailsService.getSkills({Details: $event.query})
            .pipe(takeUntil(this.componentInView))
            .subscribe(response => {
                this.skills = response;
            }, err => {
                this.utilsService.handleError(err);
            })
    }

    getSkillRatings(): void {
        this.personalDetailsService.getSkillRatings()
            .pipe(takeUntil(this.componentInView))
            .subscribe(response => {
                this.ratings = response;
            }, err => {
                this.utilsService.handleError(err);
            })
    }

    addSkill(initLoad = false): void {
        const newModel = new CareerSkillModel();
        newModel.IsNew = true;
        newModel.CareerProfile = this.careerProfile.ObjectID;
        if (initLoad && (!this.careerProfile.Skills || this.careerProfile.Skills.length === 0)) {
            this.utilsService.addMultiRefObject(newModel, this.careerProfile, 'Skills', this.createdObjs);
        } else if (!initLoad) {
            this.utilsService.addMultiRefObject(newModel, this.careerProfile, 'Skills', this.createdObjs);
        }
    }

    removeSkill(exp): void {
        this.confirmationService.confirm({
            message:'Are you sure you want to remove this skill?',
            accept: () => {
                this.utilsService.removeMultiRefObject(exp, this.careerProfile, 'Skills', this.createdObjs, this.updatedObjs, this.deletedObjs);
            }
        })
    }

    saveSkill(customRoute = null): void {
        if (this.form.invalid) {
            return this.helperService.validateAllFormFields(this.form);
        }
        this.isSaving = true;
        if (this.careerProfile.CompletedPercentageCH <= 60) {
            this.careerProfile.CurrentProfileBuilderTab = CAREER_TABS.REFERENCES;
        }
        this.personalDetailsService.saveProfileDefault(this.createdObjs, this.updatedObjs, this.deletedObjs)
            .pipe(takeUntil(this.componentInView))
            .subscribe(res => {
                this.isSaving = false;
                this.careerProfile.Skills.forEach(s => s.IsNew = false);
                if (customRoute) {
                    this.router.navigate([customRoute])
                } else {
                    this.skillsSaved.emit();
                }
            }, err => {
                this.isSaving = false;
                this.utilsService.handleError(err);
            })
    }

}
