import { Component, OnInit } from '@angular/core';
import { Router } from '@angular/router';
import { combineLatest } from 'rxjs/observable/combineLatest';
import { Subscription } from 'rxjs/Subscription';
import { AppService } from '../../app.service';
import { SEARCH, SERVICES } from '../../constants';
import { SorterModel } from '../../models/sorterModel';
import { WorkflowTemplateModel } from '../../models/workflowTemplate.model';
import { UtilsService } from '../../oneit/services/utils.service';
import { AdminPortalLayoutService } from '../admin-portal/admin-portal-layout/admin-portal-layout.service';
import { BaseComponent } from '../base/base.component';

@Component({
    selector: 'app-list-work-flows',
    templateUrl: './list-work-flows.component.html'
})
export class ListWorkFlowsComponent extends BaseComponent implements OnInit {

    subscriptions: Array<Subscription> = [];
    myTemplates: Array<WorkflowTemplateModel> = [];
    sharedTemplates: Array<WorkflowTemplateModel> = [];
    sortOptions: Array<SorterModel>;
    showLoader = false;
    processing = false;

    constructor(
        private router: Router,
        private utilsService: UtilsService,
        private appService: AppService,
        private adminPortalLayoutService: AdminPortalLayoutService
    ) {
        super(utilsService);
    }

    ngOnInit(): void {
        this.sortOptions = [
            new SorterModel('Alphabetical (A-Z)', true),
            new SorterModel('Alphabetical (Z-A)', false)
        ];
        this.showLoader = true;
        this.processing = true;

        this.startComponent();
        this.subscriptions.push(this.adminPortalLayoutService.userDataUpdated
            .subscribe(
                () => {
                    this.startComponent();
                }
            ));
    }

    startComponent(): void {

        if (!this.processing) {
            return;
        }

        this.processing = false;
        const sharedParams = {
            OnlyGlobalTemplates: true
        };
        const myParams = {
            OnlyGlobalTemplates: false
        };

        this.subscriptions.push(
            combineLatest(
                this.appService.getSearchListing(SERVICES.WORKFLOW_TEMPLATES, 'All', myParams, [], SEARCH.WORKFLOW_TEMPLATE),
                this.appService.getSearchListing(SERVICES.WORKFLOW_TEMPLATES, 'All', sharedParams, [], SEARCH.WORKFLOW_TEMPLATE)
            )
                .subscribe(
                    (response: [Array<WorkflowTemplateModel>, Array<WorkflowTemplateModel>]) => {
                        this.myTemplates = response[0].filter(wf => wf.HiringTeam ===
                            this.adminPortalLayoutService.getHiringTeamID().ObjectID);
                        this.sortSpecificTemplate('myTemplates');
                        this.sharedTemplates = response[1];
                        this.sortSpecificTemplate('sharedTemplates');
                        const val = this.myTemplates.map(m => m.ObjectID)
                            .join(',');
                        console.log(val);
                        this.showLoader = false;
                    },
                    error => {
                        this.showLoader = false;
                        this.utilsService.handleError(error);
                    }
                )
        );
    }

    sortSpecificTemplate(arrayName: string): void {
        this[arrayName] = this.utilsService.sort(this[arrayName],
            ['TemplateName'], [this.sortOptions[0].ascending ? 1 : -1], false);
    }

    sortWorkFlowTemplates($event): void {
        if (!$event.data[0]) {
            return;
        }
        const sort = $event.data[0];
        this.showLoader = true;
        this.myTemplates = this.utilsService.sort(this.myTemplates,
            ['TemplateName'], [sort.ascending ? 1 : -1], false);
        this.sharedTemplates = this.utilsService.sort(this.sharedTemplates,
            ['TemplateName'], [sort.ascending ? 1 : -1], false);
        this.showLoader = false;

    }

    onRowSelect(data: WorkflowTemplateModel): void {
        this.router.navigate(['/admin/edit-workflow-template', data.ObjectID]);
    }
}
