import { Component, EventEmitter, OnInit, Output, ViewChild } from '@angular/core';
import { BaseComponent } from '../../../base/base.component';
import { CareerProfileModel } from '../../../models/career-profile.model';
import { NgForm } from '@angular/forms';
import { PersonalDetailsService } from '../../../services/personal-details.service';
import { UtilsService } from '../../../../oneit/services/utils.service';
import { ConfirmationService } from 'primeng/api';
import { takeUntil } from 'rxjs/operators';
import { CareerSkillModel, SkillModel } from '../../../models/career-skill.model';

@Component({
    selector: 'app-skills',
    templateUrl: './skills.component.html',
    styleUrls: ['./skills.component.scss']
})
export class SkillsComponent extends BaseComponent implements OnInit {

    assocs = ['Skills.Skill'];
    skills: SkillModel[] = [];
    careerProfile = new CareerProfileModel();
    ratings = [];

    @ViewChild('form') form: NgForm;
    @Output() skillsSaved = new EventEmitter();

    constructor(
        private personalDetailsService: PersonalDetailsService,
        private utilsService: UtilsService,
        private confirmationService: ConfirmationService
    ) {
        super();
    }

    ngOnInit() {
        this.getCareerProfile();
        this.getSkillRatings();
    }

    getCareerProfile(): void {
        this.isLoading = true;
        this.personalDetailsService.getCareerProfile(this.assocs)
            .pipe(takeUntil(this.componentInView))
            .subscribe(response => {
                this.isLoading = false;
                this.careerProfile = response;
                this.utilsService.addObjsToJSONByObjectID(this.updatedObjs, this.careerProfile.Skills);
                this.addSkill(true);
                this.isLoading = false;
            }, err => {
                this.isLoading = false;
                this.utilsService.handleError(err);
            })
    }

    getSkills($event): void {
        this.personalDetailsService.getSkills({Description: $event.query})
            .pipe(takeUntil(this.componentInView))
            .subscribe(response => {
                this.skills = response;
            }, err => {
                this.utilsService.handleError(err);
            })
    }

    getSkillRatings(): void {
        this.personalDetailsService.getSkillRatings()
            .pipe(takeUntil(this.componentInView))
            .subscribe(response => {
                this.ratings = response;
            }, err => {
                this.utilsService.handleError(err);
            })
    }

    addSkill(initLoad = false): void {
        const newModel = new CareerSkillModel();
        newModel.CareerProfile = this.careerProfile.ObjectID;
        if (initLoad && (!this.careerProfile.Skills || this.careerProfile.Skills.length === 0)) {
            this.utilsService.addMultiRefObject(newModel, this.careerProfile, 'Skills', this.createdObjs);
        } else if (!initLoad) {
            this.utilsService.addMultiRefObject(newModel, this.careerProfile, 'Skills', this.createdObjs);
        }
    }

    removeSkill(exp): void {
        this.confirmationService.confirm({
            message:'Are you sure you want to remove this skill?',
            accept: () => {
                this.utilsService.removeMultiRefObject(exp, this.careerProfile, 'Skills', this.createdObjs, this.updatedObjs, this.deletedObjs);
            }
        })
    }

    saveSkill(): void {
        if (this.form.invalid) {
            return this.utilsService.showAllErrorMessages();
        }
        this.isSaving = true;
        this.personalDetailsService.saveProfile(this.createdObjs, this.updatedObjs, this.deletedObjs)
            .pipe(takeUntil(this.componentInView))
            .subscribe(res => {
                this.isSaving = false;
                this.skillsSaved.emit();
            }, err => {
                this.isSaving = false;
                this.utilsService.handleError(err);
            })
    }

}
