import { Component, OnInit } from '@angular/core';
import { Router } from '@angular/router';
import { takeUntil } from 'rxjs/operators';
import { UtilsService } from '../../../oneit/services/utils.service';
import { AssessmentBaseComponent } from '../../base/assessment-base.component';
import { DiversityProfileQuestionModel } from '../../models/diversity-profile-question.model';
import { AssessmentService } from '../../services/assessment.service';
import { ToasterService } from '../../services/toaster.service';

@Component({
    selector: 'app-diversity-profile',
    templateUrl: './diversity-profile.component.html',
    styleUrls: ['./diversity-profile.component.scss']
})
export class DiversityProfileComponent extends AssessmentBaseComponent implements OnInit {

    diversityProfiles: Array<DiversityProfileQuestionModel> = [];
    assocs = ['Answers'];
    selectedAnswers = [];

    constructor(
        private assessmentService: AssessmentService,
        private utilsService: UtilsService,
        private toasterService: ToasterService,
        private router: Router
    ) {
        super();
    }

    ngOnInit(): void {
        this.initiateDiversityProfile();
        this.setAutoSaveInterval();
    }

    initiateDiversityProfile(): void {
        this.isLoading = true;
        this.assessmentService.getDiversityProfile(this.assocs)
            .pipe(takeUntil(this.componentInView))
            .subscribe(response => {
                this.isLoading = false;
                this.autoSavedOn = new Date();
                this.diversityProfiles = response;
                this.setAnswers();
            }, err => {
                this.isLoading = false;
                this.utilsService.handleError(err);
            });
    }

    setAnswers(): void {
        this.diversityProfiles.forEach(div => {
            this.selectedAnswers[div.ObjectID] = !div.MultipleAnswers ?
                div.Answers.find(a => a.IsSelected) : div.Answers.filter(a => a.IsSelected).map(a => a.ObjectID)
        });
    }

    get isComplete(): boolean {
        return this.diversityProfiles.every(div =>
            div.MultipleAnswers ? this.selectedAnswers[div.ObjectID].length > 0 : !!this.selectedAnswers[div.ObjectID]);
    }

    exit(): void {
        this.isSaving = true;
        const answers = [];
        this.diversityProfiles.forEach(div => {
            if (div.MultipleAnswers) {
                answers.push(...this.selectedAnswers[div.ObjectID]);
            } else {
                if (this.selectedAnswers[div.ObjectID]) {
                    answers.push(this.selectedAnswers[div.ObjectID])
                }
            }
        });
        this.assessmentService.saveDiversityProfile({Answers: answers})
            .pipe(takeUntil(this.componentInView))
            .subscribe(() => {
                this.isSaving = false;
                this.toasterService.success('Diversity profile updated!');
                this.router.navigate(['/my-career-web/dashboard/home']);
            }, err => {
                this.isSaving = false;
                this.utilsService.handleError(err);
            });
    }

}
