import { ChangeDetectorRef, Component, OnInit, ViewChild } from '@angular/core';
import { NgForm } from '@angular/forms';
import { takeUntil } from 'rxjs/operators';
import { environment } from '../../../../../environments/environment';
import { AppService } from '../../../../app.service';
import { UtilsService } from '../../../../oneit/services/utils.service';
import { BaseComponent } from '../../base/base.component';
import { CareerProfileModel } from '../../models/career-profile.model';
import { HelperService } from '../../services/helper.service';
import { PersonalDetailsService } from '../../services/personal-details.service';
import { ToasterService } from '../../services/toaster.service';

@Component({
    selector: 'app-personal-details',
    templateUrl: './personal-details.component.html',
    styleUrls: ['./personal-details.component.scss']
})

export class PersonalDetailsComponent extends BaseComponent implements OnInit {

    isLoadingStatuses = false;
    isLoadingPreferences = false;
    employmentStatuses = [];
    employmentPreferences = [];
    jobTitles = [];
    careerProfile = new CareerProfileModel();
    assocs = ['Candidate.User', 'Candidate.OccupationPreference'];

    @ViewChild('form') form: NgForm;

    constructor(
        private ps: PersonalDetailsService,
        private ts: ToasterService,
        private hs: HelperService,
        private aps: AppService,
    ) {
        super();
    }

    ngOnInit() {
        this.getEmploymentStatuses();
        this.getEmploymentPreferences();
        this.getCareerProfile();
    }

    getEmploymentStatuses(): void {
        this.isLoadingStatuses = true;
        this.ps.getEmploymentStatuses()
            .pipe(takeUntil(this.componentInView))
            .subscribe(response => {
                this.employmentStatuses = response;
                this.isLoadingStatuses = false;
            }, err => {
                this.hs.handleHttpError(err);
                this.isLoadingStatuses = false;
            })
    }

    getEmploymentPreferences(): void {
        this.isLoadingPreferences = true;
        this.ps.getEmploymentPreferences()
            .pipe(takeUntil(this.componentInView))
            .subscribe(response => {
                this.employmentPreferences = response;
                this.isLoadingPreferences = false;
            }, err => {
                this.hs.handleHttpError(err);
                this.isLoadingPreferences = false;
            })
    }

    getJobTitles($event): void {
        this.ps.getJobTitles({JobTitle: $event.query})
            .pipe(takeUntil(this.componentInView))
            .subscribe(response => {
                this.jobTitles = response.filter(r => r.JobTitle);
            }, err => {
                this.hs.handleHttpError(err);
            })
    }

    getCareerProfile(): void {
        this.isLoading = true;
        this.ps.getCareerProfile(this.assocs)
            .pipe(takeUntil(this.componentInView))
            .subscribe(response => {
                this.careerProfile = response;
                this.updatedObjs[this.careerProfile.ObjectID] = this.careerProfile;
                this.updatedObjs[this.careerProfile.Candidate.ObjectID] = this.careerProfile.Candidate;
                this.updatedObjs[this.careerProfile.Candidate.User.ObjectID] = this.careerProfile.Candidate.User;
                if (this.careerProfile.Candidate.EmploymentPreference) {
                    this.careerProfile.Candidate.EmploymentPreference = this.careerProfile.Candidate.EmploymentPreference.map(p => p.Value);
                }
                this.ps.careerProfileUpdated = this.careerProfile;
                this.isLoading = false;
            }, err => {
                this.hs.handleHttpError(err);
                this.isLoading = false;
            })
    }

    updateProfile(): void {
        this.hs.validateAllFormFields(this.form);
        if (this.form.invalid) {
            return ;
        }
        this.careerProfile.Candidate.User.UserName = this.careerProfile.Candidate.User.Email;
        this.isSaving = true;
        this.ps.saveProfile(this.createdObjs, this.updatedObjs, this.deletedObjs)
            .pipe(takeUntil(this.componentInView))
            .subscribe(res => {
                this.isSaving = false;
                this.hs.oneItHttpResponse(res);
                this.ps.careerProfileUpdated = this.careerProfile;
                this.ts.success('Profile updated.');
            }, err => {
                this.hs.handleHttpError(err);
                this.isSaving = false;
            })
    }

    checkMobilePattern() {
        if (this.careerProfile.Mobile && this.careerProfile.Mobile.length < 11) {
            this.careerProfile.Mobile = '';
        }
    }

    changedAddress(): void {
        this.careerProfile.GoogleAddress = ' ';
    }

    updateAddress($event): void {
        this.careerProfile.GoogleAddress = $event.formatted;
    }

    // @CodeReview : G002 : can reuse alreayd created method 
    connectWithSocial(type): void {
        const params = {
            environment: environment.envName,
            nextPage: window.location.href.split('/#/')[0] + '/#/my-career-web/dashboard/home'
        };
        const svc = type === 'LINKEDIN' ? 'LoginWithLinkedIn' : 'LoginWithGoogle';
        this.aps.socicalMediaLogin(svc, params)
            .pipe(takeUntil(this.componentInView))
            .subscribe(data => {
                    this.isLoading = false;
                    if (data.destination) {
                        window.location.href = data.destination;
                        return;
                    }
                },
                err => {
                    this.hs.handleHttpError(err);
                }
            )
    }

}
