import { Component, EventEmitter, Input, NgZone, OnInit, Output, ViewChild } from '@angular/core';
import { NgForm } from '@angular/forms';
import { ActivatedRoute, Params, Router } from '@angular/router';
import { Editor } from 'primeng/editor';
import { OverlayPanel } from 'primeng/overlaypanel';
import { Subscription } from 'rxjs/Subscription';
import { AppService } from '../../app.service';
import { SERVICES } from '../../constants';
import { MessageTemplateModel } from '../../models/message-template.model';
import { PlaceholderOptionsEnum } from '../../models/placeholder-options.enum';
import { TemplateTypeEnum } from '../../models/template-type.enum';
import { SaveService } from '../../oneit/services/save.service';
import { UtilsService } from '../../oneit/services/utils.service';
import { AdminPortalLayoutService } from '../admin-portal/admin-portal-layout/admin-portal-layout.service';
import { BaseComponent } from '../base/base.component';

@Component({
    selector: 'app-edit-message-template',
    templateUrl: './edit-message-template.component.html'
})
export class EditMessageTemplateComponent extends BaseComponent implements OnInit {

    @Input() objectID: string;
    // tslint:disable-next-line:prefer-output-readonly
    @Output() closeModal: EventEmitter<any> = new EventEmitter<any>();

    subscriptions: Array<Subscription> = [];
    @ViewChild('form') form: NgForm;
    @ViewChild('messageContent') messageContentEditor: Editor;
    @ViewChild('contentLinkModal') contentLinkModal: OverlayPanel;
    createdObjs;
    updatedObjs;
    deletedObjs;

    messageTemplate = new MessageTemplateModel();
    messageTemplateTypes: Array<TemplateTypeEnum> = [];
    subjectPlaceholderOptions: Array<PlaceholderOptionsEnum> = [];
    contentPlaceholderOptions: Array<PlaceholderOptionsEnum> = [];
    ContentPlaceholderOptionsEnum: Array<PlaceholderOptionsEnum> = [];
    subjectPlaceholder: PlaceholderOptionsEnum;
    contentPlaceholder: PlaceholderOptionsEnum;
    contentCursorPoint;
    temporaryMessageContent = '';
    linkText = '';
    options = {minimumResultsForSearch: Infinity};
    templateTypeId = 0;
    showLoader = false;
    editMode = false;

    constructor(
        private router: Router,
        private activatedRoute: ActivatedRoute,
        private utilsService: UtilsService,
        private saveService: SaveService,
        private appService: AppService,
        private adminPortalLayoutService: AdminPortalLayoutService,
        public ngZone: NgZone
    ) {
        super(utilsService);
    }

    ngOnInit(): void {
        this.createdObjs = {};
        this.updatedObjs = {};
        this.deletedObjs = {};
        this.showLoader = true;

        this.utilsService.resetCounter();
        this.getPlaceholderOptions();

        if (this.adminPortalLayoutService.userData) {
            this.initComponent();
        } else {
            this.subscriptions.push(this.adminPortalLayoutService.userDataUpdated.first()
                .subscribe(
                    () => {
                        this.initComponent();
                    }
                ));
        }
    }

    initComponent(): void {
        this.subscriptions.push(this.appService.getMessageTemplateTypeEnum()
            .subscribe(
                data => {
                    this.messageTemplateTypes = data;

                    this.subscriptions.push(this.activatedRoute.params
                        .subscribe(
                            (params: Params) => {

                                if (this.templateCreationFromWorkflow()) {
                                    this.messageTemplate.HiringTeam = this.adminPortalLayoutService.getHiringTeamID().ObjectID;

                                    if (this.messageTemplateTypes) {
                                        this.messageTemplate.TemplateType = this.messageTemplateTypes[0];
                                    }

                                    this.utilsService.createObject(this.messageTemplate, this.createdObjs);
                                    this.setInputListener();

                                    this.showLoader = false;
                                } else if (this.templateUpdateFromWorkflow() || (params.id && params.id !== '0')) {
                                    this.editMode = true;

                                    this.getMessageTemplateByID(this.templateUpdateFromWorkflow() ? this.objectID : params.id);
                                } else {
                                    this.messageTemplate.HiringTeam = this.adminPortalLayoutService.getHiringTeamID().ObjectID;

                                    if (this.messageTemplateTypes) {
                                        this.messageTemplate.TemplateType = this.messageTemplateTypes[0];
                                    }

                                    this.utilsService.createObject(this.messageTemplate, this.createdObjs);
                                    this.setInputListener();

                                    this.showLoader = false;
                                }

                            }
                        ));
                },
                error => {
                    this.utilsService.handleError(error);
                }
            )
        );
    }

    showSaveBtn(): boolean {
        return this.messageTemplate && this.messageTemplate.ObjectID ?
            (this.messageTemplate.HiringTeam === this.adminPortalLayoutService.getHiringTeamID().ObjectID ||
                !this.editMode) : false;
    }

    showSaveIntoAllTemplates(): boolean {
        return !!this.messageTemplate && !!this.messageTemplate.ObjectID &&
            (!this.messageTemplate.IsSystemGenerated && !!this.objectID && this.editMode);
    }

    showSaveCopyBtn(): boolean {
        if (this.messageTemplate && this.messageTemplate.ObjectID) {
            return this.messageTemplate.HiringTeam === null;
        }

        return false;
    }

    setInputListener(): void {
        let editElement;
        setTimeout(() => {

            if (this.emailTemplateType()) {
                editElement = this.messageContentEditor.getQuill().container;

                if (editElement) {
                    const editor = editElement.getElementsByClassName('ql-editor');

                    if (editor[0]) {
                        editor[0].classList.add('form-control');
                    }

                    ['keyup', 'click'].forEach(event => {
                        editElement.addEventListener(event, () => {
                            this.contentCursorPoint = this.messageContentEditor.getQuill()
                                .getSelection().index;
                        });
                    });
                }

            } else {
                editElement = document.getElementById('messageContentTxtAreaId');

                if (editElement) {
                    ['keyup', 'click'].forEach(event => {
                        editElement.addEventListener(event, () => {
                            this.contentCursorPoint = this.getCursorPos(editElement);
                        });
                    });
                }

            }

        }, 0);
    }

    addAnchorText(show: boolean): void {

        if (!show || this.linkText === '') {
            this.contentLinkModal.hide();

            return;
        }

        const delta = {
            ops: [
                {retain: this.contentCursorPoint > 0 ? this.contentCursorPoint : this.contentCursorPoint + 1},
                {insert: `\${Placeholder:${this.contentPlaceholder.Placeholder}}`, attributes: {link: this.linkText}}
            ]
        };

        if (this.contentCursorPoint === 0) {
            this.messageContentEditor.getQuill()
                .insertText(this.contentCursorPoint, ' ');
            this.contentCursorPoint++;
        }

        try {
            this.messageContentEditor.getQuill()
                .updateContents(delta);
        } catch (e) {
            this.utilsService.handleError(e);
        }

        this.linkText = '';
        this.contentLinkModal.hide();
    }

    assignContent(): void {
        const editor = this.messageContentEditor.getQuill().container;
        if (editor) {
            const editElement = editor.getElementsByClassName('ql-editor')[0];
            this.messageTemplate.MessageContent = editElement.innerHTML;
        }
    }

    addPlaceholderInMessageContent($event): void {

        if (!$event) {
            return;
        }

        this.contentPlaceholder = $event;

        if (!this.contentPlaceholder || !this.contentPlaceholder.Value) {
            return;
        }

        if (typeof this.contentCursorPoint !== 'number') {
            return;
        }

        if (this.emailTemplateType()) {
            if (this.contentPlaceholder.IsLink) {
                const event = new Event('build');
                this.contentLinkModal.toggle(event, this.messageContentEditor.getQuill());
            } else {
                this.messageContentEditor.getQuill()
                    .insertText(this.contentCursorPoint, `\${Placeholder:${this.contentPlaceholder.Placeholder}}`);
            }
        } else {
            const textarea: HTMLTextAreaElement = document.getElementById('messageContentTxtAreaId') as HTMLTextAreaElement;

            this.setValueAndFocus(textarea, 'messageContentTxtArea', `\${Placeholder:${this.contentPlaceholder.Placeholder}}`,
                'MessageContent');
        }

        this.resetSelect('contentPlaceholderOptions');
    }

    showPlainText(HTMLTitle: string): string {
        const temp = document.getElementById('showPlainText');
        temp.innerHTML = HTMLTitle;
        const title = temp.innerText.trim();
        temp.innerHTML = '';

        return title;
    }

    storeMessageContent($event): void {

        if (!$event) {
            return;
        }

        this.messageTemplate.TemplateType = $event;

        if (this.messageTemplate.MessageContent !== undefined) {
            if (this.smsTemplateType()) {
                this.temporaryMessageContent = this.messageTemplate.MessageContent;
                this.messageTemplate.MessageContent = this.showPlainText(this.temporaryMessageContent);
            } else {
                this.messageTemplate.MessageContent = this.temporaryMessageContent !== '' ? this.temporaryMessageContent
                    : this.messageTemplate.MessageContent;
            }
        }

        if (this.ContentPlaceholderOptionsEnum) {
            if (this.emailTemplateType()) {
                this.contentPlaceholderOptions = this.ContentPlaceholderOptionsEnum.filter(option => option.ShowForContent ||
                    option.Value === '-');
            } else {
                this.contentPlaceholderOptions = this.ContentPlaceholderOptionsEnum
                    .filter(option => option.ShowForContent && !option.IsLink || option.Value === '-');
            }
        }

        this.setInputListener();
    }

    addPlaceholderInSubject($event): void {
        this.subjectPlaceholder = $event;
        if (this.emailTemplateType() && this.subjectPlaceholder && this.subjectPlaceholder.Value) {
            const input: HTMLInputElement = document.getElementById('subjectInputId') as HTMLInputElement;
            this.setValueAndFocus(input, 'subjectInput', `\${Placeholder:${this.subjectPlaceholder.Placeholder}}`, 'Subject');
        }
        this.resetSelect('subjectPlaceholderOptions');
    }

    resetSelect(key: string): void {
        const temp = this[key];
        this[key] = [];
        setTimeout(() => (this[key] = temp), 0);
    }

    getCursorPos(input): number {

        if ('selectionStart' in input && document.activeElement === input) {
            return input.selectionStart;
        }

        return -1;
    }

    getMessageTemplateByID(id): void {
        this.showLoader = true;
        this.createdObjs = {};
        this.updatedObjs = {};
        this.deletedObjs = {};

        this.utilsService.resetCounter();
        this.utilsService.clearErrorMessages();

        this.subscriptions.push(this.appService.getMessageTemplateByID(id, [])
            .subscribe(
                data => {
                    this.messageTemplate = data;
                    this.updatedObjs[this.messageTemplate.ObjectID] = this.messageTemplate;
                    this.templateTypeId =
                        this.messageTemplateTypes.find(elem => elem.Value === this.messageTemplate.TemplateType.Value).id;
                    this.showLoader = false;
                    setTimeout(() => this.setInputListener(), 0);
                },
                error => {
                    this.showLoader = false;
                    this.utilsService.handleError(error);
                }
            )
        );
    }

    // If new message template is being created from 'EditWorkFlowComponent'
    templateCreationFromWorkflow(): boolean {
        return this.objectID && this.objectID.indexOf('NEW') !== -1;
    }

    // If a message template is being updated from 'EditWorkFlowComponent'
    templateUpdateFromWorkflow(): boolean {
        return this.objectID && this.objectID.indexOf('NEW') === -1;
    }

    cancelClicked(): void {
        // If new message template is being created from 'EditWorkFlowComponent'
        if (this.templateCreationFromWorkflow() || this.templateUpdateFromWorkflow()) {
            this.closeModal.emit(-1);
        } else {
            this.router.navigate(['/admin/list-message-templates']);
        }
    }

    saveMessageTemplate(saveCopy = false): void {
        this.assignContent();
        this.utilsService.clearErrorMessages();
        const service = `svc/${saveCopy ? SERVICES.SAVE_MESSAGE_AS_COPY : SERVICES.SAVE}`;

        if (this.form.invalid) {
            this.utilsService.showAllErrorMessages();
        } else {
            this.showLoader = true;
            this.subscriptions.push(this.saveService.saveObjects(service, this.createdObjs, this.updatedObjs, this.deletedObjs)
                .subscribe(
                    data => {
                        this.utilsService.handleSuccess();

                        if (this.templateCreationFromWorkflow()) {
                            let id = this.messageTemplate.ObjectID;

                            if (data.created[id]) {
                                id = data.created[id];
                            }

                            this.closeModal.emit(id);
                        } else if (this.templateUpdateFromWorkflow()) {
                            this.closeModal.emit(this.messageTemplate.ObjectID);
                        } else {
                            this.router.navigate(['/admin/list-message-templates']);
                        }

                    },
                    error => {
                        this.showLoader = false;
                        this.utilsService.handleError(error);
                    }
                )
            );
        }
    }

    getPlaceholderOptions(): void {
        this.subscriptions.push(this.appService.getPlaceholderOptionsEnum()
            .subscribe(options => {
                this.ContentPlaceholderOptionsEnum = options;
                this.ContentPlaceholderOptionsEnum.filter(option => option.Value === null)
                    .map(val => {
                        val.Value = '-';

                        return val;
                    });
                this.contentPlaceholderOptions =
                    this.ContentPlaceholderOptionsEnum.filter(option => option.ShowForContent || option.Value === '-');
                this.subjectPlaceholderOptions =
                    this.ContentPlaceholderOptionsEnum.filter(option => option.ShowForSubject || option.Value === '-');
            })
        );
    }

    emailTemplateType(): boolean {
        return this.messageTemplate && this.messageTemplate.TemplateType ?
            this.messageTemplate.TemplateType.Value === 'EMAIL' :
            false;
    }

    smsTemplateType(): boolean {
        return this.messageTemplate && this.messageTemplate.TemplateType ?
            this.messageTemplate.TemplateType.Value === 'SMS' :
            false;
    }

    setValueAndFocus(element, elementId, placeholder, modelKey): void {
        const start = element.selectionStart;
        const end = element.selectionEnd;
        this.messageTemplate[modelKey] = `${element.value.substring(0, start)}${placeholder}${element.value.substring(end)}`;

        this.focusField(elementId);

        setTimeout(() => {
            element.selectionStart = element.selectionEnd = end + placeholder.length;
        }, 0);
    }

    focusField(elementId: string): void {
        setTimeout(() => {
            try {
                document.getElementById(`${elementId}Id`)
                    .focus();
                document.getElementById(`${elementId}Id`)
                    .getElementsByTagName('input')[0]
                    .focus();
            } catch (e) {
                // ignore this.
            }
        }, 0);
    }

// tslint:disable-next-line:max-file-line-count
}
