import { Component, EventEmitter, Input, OnInit, Output, ViewChild } from '@angular/core';
import { NgForm } from '@angular/forms';
import { ActivatedRoute, NavigationEnd, Router } from '@angular/router';
import { MenuItem } from 'primeng/api';
import { Editor } from 'primeng/editor';
import { FileUpload } from 'primeng/fileupload';
import { Subscription } from 'rxjs/Subscription';
import { environment } from '../../../environments/environment';

import { AppService } from '../../app.service';
import { ASSOCS } from '../../constants';
import { ChatAttachmentModel } from '../../models/chat-attachment.model';
import { JobApplicationModel } from '../../models/job-application.model';
import { MessageModel } from '../../models/message.model';
import { SentViaEnum } from '../../models/sent-via.enum';
import { SaveService } from '../../oneit/services/save.service';
import { UtilsService } from '../../oneit/services/utils.service';
import { AdminPortalLayoutService } from '../admin-portal/admin-portal-layout/admin-portal-layout.service';
import { BaseComponent } from '../admin-portal/base/base.component';
import { ApplicantPortalLayoutService } from '../applicant-portal/applicant-portal-layout/applicant-portal-layout.service';
import { MessageEngineApplicantViewService } from './message-engine-applicant-view.service';

@Component({
    selector: 'app-message-engine-applicant-view',
    templateUrl: './message-engine-applicant-view.component.html',
    styleUrls: ['./message-engine-applicant-view.component.scss']
})
export class MessageEngineApplicantViewComponent extends BaseComponent implements OnInit {

    @Input() embedded = false;
    @Input() jobId = null;
    // tslint:disable-next-line:prefer-output-readonly
    @Output() minimize: EventEmitter<any> = new EventEmitter<any>();
    @ViewChild('message') messageEditor: Editor;
    @ViewChild('form') form: NgForm;
    @ViewChild('uploader') uploader: FileUpload;
    sentViaMenuItems: Array<MenuItem>;
    createdObjs = {};
    updatedObjs = {};
    deletedObjs = {};

    jobApplication = new JobApplicationModel();
    subscriptions: Array<Subscription> = [];
    sentViaEnum: Array<SentViaEnum> = [];
    Message = new MessageModel();
    apiUrl = environment.baseUrl;
    loggedInId = null;
    showLoader = false;
    previousUrl: string;

    constructor(
        private router: Router,
        private activatedRoute: ActivatedRoute,
        private utilsService: UtilsService,
        private saveService: SaveService,
        private appService: AppService,
        private messageEngineApplicantViewService: MessageEngineApplicantViewService,
        private adminPortalLayoutService: AdminPortalLayoutService,
        private applicantPortalLayoutService: ApplicantPortalLayoutService
    ) {
        super(utilsService);
        // In order to get previous url for navigation
        router.events
            .filter(event => event instanceof NavigationEnd)
            .subscribe((e: any) => {
                this.previousUrl = e.url;
            });
    }

    ngOnInit(): void {

        this.getSetViaOptions();
        // For Hiring Team or Applicant
        const service = this.embedded ? this.adminPortalLayoutService : this.applicantPortalLayoutService;
        if (service.userData) {
            this.initComponent(service);
        } else {
            this.subscriptions.push(service.userDataUpdated.first()
                .subscribe(() => {
                    this.initComponent(service);
                }));
        }
    }

    getSetViaOptions(): void {
        this.subscriptions.push(
            this.appService.getTypes('SentVia')
                .subscribe((data: Array<SentViaEnum>) => {
                    this.sentViaEnum = data;
                    this.sentViaMenuItems = this.sentViaEnum.map(elem =>
                        ({
                            label: elem.Description, command: () => this.Message.SentVia = elem
                        }));

                })
        );
    }

    initComponent(service: AdminPortalLayoutService | ApplicantPortalLayoutService): void {
        if (service instanceof AdminPortalLayoutService) {
            this.loggedInId = service.userData.CompanyUser.User;
            // jobId should come from @Input (Admin Portal)
        } else {
            this.jobId = service.jobId;
            this.loggedInId = service.userData.UserID;
        }

        if (this.jobId) {
            this.getEngineApplicantViewByID();
        } else {
            // Should never come to this block
            console.error('JobId not found.');
        }
    }

    getEngineApplicantViewByID(): void {
        this.showLoader = true;
        this.createdObjs = {};

        const assocs =
            [ASSOCS.JOB, [ASSOCS.JOB, ASSOCS.HIRING_TEAM].join('.'), ASSOCS.MESSAGES, [ASSOCS.MESSAGES, ASSOCS.SENT_BY].join('.'),
                [ASSOCS.MESSAGES, ASSOCS.ATTACHMENTS].join('.'), [ASSOCS.CANDIDATE, ASSOCS.USER].join('.')];

        this.subscriptions.push(this.messageEngineApplicantViewService.getEngineApplicantViewByID(this.jobId, assocs)
            .subscribe(
                data => {
                    this.jobApplication = data;
                    this.jobApplication.Messages = this.jobApplication.Messages || [];
                    this.utilsService.sort(this.jobApplication.Messages, ['ObjectCreated'], [1]);

                    this.jobApplication.Messages.forEach(msg => {
                        msg.ObjectCreated = new Date(msg.ObjectCreated);
                    });

                    this.createMessage();

                    this.showLoader = false;
                    setTimeout(() => {
                        const msgWrapper = document.getElementsByClassName('messages-wrapper')[0];
                        msgWrapper.scrollTop = msgWrapper.scrollHeight;
                    }, 0);
                },
                error => {
                    this.showLoader = false;
                    this.utilsService.handleError(error);
                }
            )
        );
    }

    /**
     * This function sets isRead property of all messages of this job application to true
     */
    markAsRead(): void {
        this.showLoader = true;
        const params = {
            formParams: {
                JobApplication: this.jobApplication.ObjectID
            }
        };
        this.appService.processObjects('MarkAsRead', params)
            .subscribe(
                () => this.showLoader = false,
                err => {
                    this.utilsService.handleError(err);
                    this.showLoader = false;
                }
            );
    }

    saveEngineApplicantView(): void {
        if (this.showLoader) {
            return;
        }

        if (!this.Message.SentVia) {
            this.Message.SentVia = this.sentViaEnum[2] ? this.sentViaEnum[2] : this.sentViaEnum[0];
        }

        this.showLoader = true;
        this.subscriptions.push(this.saveService.saveObjectsWithDefaultSvc(this.createdObjs, this.updatedObjs, this.deletedObjs)
            .subscribe(
                () => {
                    this.utilsService.handleSuccess();
                    this.showLoader = false;
                    this.getEngineApplicantViewByID();
                },
                error => {
                    this.showLoader = false;
                    this.utilsService.handleError(error);
                }
            )
        );
    }

    /**
     * Creates a new object of ChatAttachment and assigns the FileToken and Name to it.
     * Moreover, pushes it to the current Message's Attachments Array
     * @param event Browser Event
     */
    onUpload(event: any): void {
        const chatAttachment = new ChatAttachmentModel();
        this.utilsService.createObject(chatAttachment, this.createdObjs);
        chatAttachment.File = {
            FileToken: JSON.parse(event.xhr.response).files[0].token,
            Name: JSON.parse(event.xhr.response).files[0].fileName,
            Index: this.Message.Attachments.length
        };
        chatAttachment.Message = this.Message;
        this.Message.Attachments.push(chatAttachment);
    }

    removeAttachment(file): void {
        this.utilsService.removeMultiRefObject(file, this.Message, 'Attachments', this.createdObjs,
            this.updatedObjs, this.deletedObjs);
    }

    /**
     * It simply creates a new object and assigns the default values to it.
     */
    createMessage(): void {
        this.Message = new MessageModel();
        this.Message.SentBy = this.loggedInId;
        this.Message.Attachments = [];
        this.Message.JobApplication = this.jobApplication;
        this.utilsService.createObject(this.Message, this.createdObjs);
    }

    /**
     * Helper functions
     */

    triggerUpload(): void {
        // @ts-ignore
        this.uploader.el.nativeElement.getElementsByTagName('input')[0]
            .click();
    }

    sendCriteria(): boolean {
        if (this.showLoader || !this.Message.Attachments) {
            return true;
        }

        return (this.Message.MessageContent === '' || !this.Message.MessageContent) && !this.Message.Attachments.length;
    }
}
