import { Component, OnInit } from '@angular/core';
import { Router, Params, ActivatedRoute} from '@angular/router';
import { combineLatest } from 'rxjs/observable/combineLatest';
import { Subscription } from 'rxjs/Subscription';
import { AppService } from '../../../app.service';
import { SEARCH, SERVICES } from '../../../constants';
import { SorterModel } from '../../../models/sorter-model';
import { WorkflowTemplateModel } from '../../../models/workflow-template.model';
import { UtilsService } from '../../../oneit/services/utils.service';
import { AdminPortalLayoutService } from '../admin-portal-layout/admin-portal-layout.service';
import { BaseComponent } from '../base/base.component';

@Component({
    selector: 'app-list-work-flows',
    templateUrl: './list-work-flows.component.html'
})
export class ListWorkFlowsComponent extends BaseComponent implements OnInit {

    subscriptions: Array<Subscription> = [];
    myTemplates: Array<WorkflowTemplateModel> = [];
    sharedTemplates: Array<WorkflowTemplateModel> = [];
    sortOptions: Array<SorterModel>;
    showLoader = false;
    isGlobalHT = false;

    constructor(
        private router: Router,
        private activatedRoute: ActivatedRoute,
        private utilsService: UtilsService,
        private appService: AppService,
        private adminPortalLayoutService: AdminPortalLayoutService
    ) {
        super(utilsService);
    }

    ngOnInit(): void {
        this.subscriptions.push(this.activatedRoute.params
            .subscribe(
                (params: Params) => {
                    if ((params.global && params.global === 'global')) {
                        this.isGlobalHT = true;
                    }
                }
            ));

        this.sortOptions = [
            new SorterModel('Alphabetical (A-Z)', true),
            new SorterModel('Alphabetical (Z-A)', false)
        ];
        this.showLoader = true;

        if (this.adminPortalLayoutService.userData || this.isGlobalHT) {
            this.initComponent();
        } else {
            this.subscriptions.push(
                this.adminPortalLayoutService.userDataUpdated.first()
                    .subscribe(
                        () => this.initComponent()
                    )
            );
        }

    }

    initComponent(): void {
        const sharedParams = {
            OnlyGlobalTemplates: true
        };

        if(this,this.isGlobalHT) {
            this.subscriptions.push(
                combineLatest(
                    this.appService.getSearchListing(SERVICES.WORKFLOW_TEMPLATES, 'All', sharedParams, [], SEARCH.WORKFLOW_TEMPLATE)
                )
                    .subscribe(
                        (response: [Array<WorkflowTemplateModel>]) => {
                            this.sharedTemplates = response[0];
                            this.sortSpecificTemplate('sharedTemplates');
                            this.showLoader = false;
                        },
                        error => {
                            this.showLoader = false;
                            this.utilsService.handleError(error);
                        }
                    )
            );
        } else {
            const myParams = {
                OnlyGlobalTemplates: false,
                HiringTeam: this.adminPortalLayoutService.getHiringTeamID().ObjectID
            };

            this.subscriptions.push(
                combineLatest(
                    this.appService.getSearchListing(SERVICES.WORKFLOW_TEMPLATES, 'All', myParams, [], SEARCH.WORKFLOW_TEMPLATE),
                    this.appService.getSearchListing(SERVICES.WORKFLOW_TEMPLATES, 'All', sharedParams, [], SEARCH.WORKFLOW_TEMPLATE)
                )
                    .subscribe(
                        (response: [Array<WorkflowTemplateModel>, Array<WorkflowTemplateModel>]) => {
                            this.myTemplates = response[0];
                            this.sharedTemplates = response[1];
    
                            this.sortSpecificTemplate('myTemplates');
                            this.sortSpecificTemplate('sharedTemplates');
    
                            this.showLoader = false;
                        },
                        error => {
                            this.showLoader = false;
                            this.utilsService.handleError(error);
                        }
                    )
            );
        }
    }

    sortSpecificTemplate(arrayName: string): void {
        this[arrayName] = this.utilsService.sort(this[arrayName],
            ['TemplateName'], [this.sortOptions[0].ascending ? 1 : -1], false);
    }

    sortWorkFlowTemplates($event): void {

        if (!$event) {
            return;
        }

        this.showLoader = true;
        this.myTemplates = this.utilsService.sort(this.myTemplates,
            ['TemplateName'], [$event.ascending ? 1 : -1], false);
        this.sharedTemplates = this.utilsService.sort(this.sharedTemplates,
            ['TemplateName'], [$event.ascending ? 1 : -1], false);
        this.showLoader = false;
    }

    onRowSelect(data: WorkflowTemplateModel): void {
        this.router.navigate([this.isGlobalHT ? '/edit-workflow-template/global' : '/admin/edit-workflow-template', data.ObjectID]);
    }

    onNewTemplate(): void {
        this.router.navigate([this.isGlobalHT ? '/edit-workflow-template/global' : '/admin/edit-workflow-template']);
    }
}
