package performa.utils;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Properties;
import java.util.Set;
import javax.mail.BodyPart;
import javax.mail.Folder;
import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.Multipart;
import javax.mail.Part;
import javax.mail.Session;
import javax.mail.Store;
import javax.mail.UIDFolder;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeMessage;
import javax.mail.search.AndTerm;
import javax.mail.search.ComparisonTerm;
import javax.mail.search.ReceivedDateTerm;
import javax.mail.search.SearchTerm;
import javax.mail.search.SentDateTerm;
import oneit.appservices.config.ConfigMgr;
import oneit.components.InitialisationParticipant;
import oneit.components.ParticipantInitialisationContext;
import oneit.email.EmailFetcher;
import oneit.logging.LogLevel;
import oneit.logging.LogMgr;
import oneit.logging.LoggingArea;
import oneit.objstore.FileBinaryContent;
import oneit.objstore.ObjectTransaction;
import oneit.objstore.StorageException;
import oneit.objstore.services.TransactionServices;
import oneit.objstore.services.TransactionTask;
import oneit.utils.DateDiff;
import oneit.utils.IOUtils;
import oneit.utils.InitialisationException;
import oneit.utils.NestedException;
import oneit.utils.StringUtils;
import oneit.utils.parsers.FieldException;
import performa.orm.Attachment;
import performa.orm.EmailMessage;
import performa.orm.Job;


public class PerformaEmailFetcher implements Runnable, InitialisationParticipant
{
    private static final String         SERVER_NAME             =   ConfigMgr.getKeyfileString("imap.server.name");
    private static final String         SERVER_PORT             =   ConfigMgr.getKeyfileString("imap.server.port");
    private static final String         SERVER_PROTOCOL         =   ConfigMgr.getKeyfileString("imap.server.protocol", "imap");
    private static final String         ACC_USER_NAME           =   ConfigMgr.getKeyfileString("imap.email.acc.username");
    private static final String         ACC_PASSWORD            =   ConfigMgr.getKeyfileString("imap.email.acc.password");
    private static final boolean        SYNC_ALL_EXISTING_EMAILS    =   ConfigMgr.getKeyfileBoolean("sync.all.existing.emails", false);
    private static final Integer        SYNC_EMAILS_SINCE_MINUTES   =   ConfigMgr.getKeyfileInt("sync.emails.since.minutes", 90);   //Applicable only when SYNC_ALL_EXISTING_EMAILS is false.
    public  static final LoggingArea    LOG =   LoggingArea.createLoggingArea("PerformaEmailFetcher");

    @Override
    public void run() 
    {
        LogMgr.log(LOG, LogLevel.PROCESSING1, "Inside Run of PerformaEmailFetcher");
        
        List<String>    visitedNodes    =   new ArrayList<>();
        
        runEmailFetcher(visitedNodes);
    }

    public static void runEmailFetcher(List<String> visitedNodes)
    {
        LogMgr.log(LOG, LogLevel.PROCESSING1, "Executing email fetcher Name:", SERVER_NAME, " Port:"+ SERVER_PORT, " Protocol:"+ SERVER_PROTOCOL, " UserName:", ACC_USER_NAME);
        LogMgr.log(LOG, LogLevel.PROCESSING1, "Email syncing details:: Sync All:", SYNC_ALL_EXISTING_EMAILS, " Sync from last:"+ SYNC_EMAILS_SINCE_MINUTES);
        
        if(StringUtils.subBlanks(SERVER_NAME) != null && StringUtils.subBlanks(SERVER_PORT) != null && 
            StringUtils.subBlanks(ACC_USER_NAME) != null && StringUtils.subBlanks(ACC_PASSWORD) != null)
        {
            final   Object[]  dummyContainer  =   new Object[1];
            Store   store   =   null;
                    
            try
            {
                TransactionServices.run(new TransactionTask()
                {
                    @Override
                    public void run(ObjectTransaction objTran) throws FieldException, StorageException
                    {
                        EmailMessage[]  emailMessages   =   EmailMessage.searchAll(objTran); 
                        
                        dummyContainer[0]   =   EmailMessage.pipesEmailMessage(emailMessages).toMessageId().uniqueVals();
                    }
                });
                
                Set<String> messageIDs  =   new HashSet();
                
                if(dummyContainer[0] != null)
                {
                    messageIDs  =   (Set<String>)dummyContainer[0];
                }
                
                Session session =   Session.getInstance(getEMailProperties());

                store   =   session.getStore(SERVER_PROTOCOL);
                store.connect(SERVER_NAME, ACC_USER_NAME, ACC_PASSWORD);

                LogMgr.log(LOG, LogLevel.PROCESSING1, "Mail connection done successfully");
                
                Folder[] folders = store.getDefaultFolder().list("*");
                
                for (Folder folder : folders) 
                {
                    if ((folder.getType() & Folder.HOLDS_MESSAGES) != 0) 
                    {
                        LogMgr.log(LOG, LogLevel.PROCESSING1, "Processing folder " + folder.getFullName());
                        visitedNodes.add(folder.getFullName());
                        
                        processFolder(folder, messageIDs, visitedNodes);
                    }
                    else
                    {
                        LogMgr.log(LOG, LogLevel.PROCESSING1, "Skipping folder " + folder.getFullName());
                    }
                }
                
                LogMgr.log(LOG, LogLevel.PROCESSING1, "Email fetcher batch completed successfully.");
            }
            catch (Exception e)
            {
                LogMgr.log(LOG, LogLevel.SYSTEMERROR1, e, "Email Fetcher : error occurred");
            }
            finally
            {
                try
                {
                    if (store != null)
                    {
                        store.close();
                    }
                }
                catch (Exception e2)
                {
                    LogMgr.log(LOG, LogLevel.BUSINESS2, "Email Fetcher : Problem closing email store", e2);
                }
            }
        }
        else
        {
            LogMgr.log(LOG, LogLevel.PROCESSING1, "Email fetcher batch skipped as server propertied not defined.");
        }
    }
    
    private static Properties getEMailProperties()
    {
        Properties  properties  =   System.getProperties();

        // Mail-server-properties will be configured based on given emailServerProtocol i.e. imap(s) or pop3(s)
        properties.put("mail.store.protocol", SERVER_PROTOCOL);
        properties.put("mail." + SERVER_PROTOCOL +".host", SERVER_NAME);
        properties.put("mail." + SERVER_PROTOCOL +".port", SERVER_PORT);
        properties.put("mail." + SERVER_PROTOCOL +".starttls.enable", "true");

        return properties;
    }
    
    private static void processFolder(Folder folder, Set<String> messageIDs, List<String> visitedNodes) throws Exception
    {
        LogMgr.log(LOG, LogLevel.PROCESSING1, "processFolder called for ", folder != null ? folder.getName() : "");
        
        if(folder != null)
        {
            try
            {
                readMessagesInFolder(folder, messageIDs);

                for(Folder subfolder : folder.list())
                {
                    visitedNodes.add(subfolder.getFullName());
                    processFolder(subfolder, messageIDs, visitedNodes);
                }
            }
            catch (Exception e)
            {
                LogMgr.log(LOG, LogLevel.SYSTEMERROR1, e, "Email Fetcher : error occurred");
                throw e;
            }
            finally
            {
                try
                {
                    folder.close(true);
                }
                catch (Exception ex2)
                {
                    LogMgr.log(LOG, LogLevel.BUSINESS2, "Email Fetcher : Problem closing email store", ex2);
                }
            }
        }
        
        LogMgr.log(LOG, LogLevel.PROCESSING1, "processFolder completed for ", folder != null ? folder.getName() : "");
    }
    
    private static void readMessagesInFolder(Folder folder, Set<String> messageIDs) throws MessagingException
    {
        LogMgr.log(LOG, LogLevel.PROCESSING1, "readMessagesInFolder called for ", folder.getName());
        
        UIDFolder   uidFolder   =   (UIDFolder)folder;
        
        try
        {
            folder.open(Folder.READ_ONLY);
        }
        catch(Exception ex)
        {
            LogMgr.log(LOG, LogLevel.PROCESSING1, "Skipping messages for folder ", folder.getName(), " as unable to open same.");
            return;
        }
        
        Message[]   messages;
        SearchTerm  searchTerm  =   getSearchTerm();
        
        if(searchTerm == null)
        {
            messages    =   folder.getMessages();
        }
        else
        {
            messages    =   folder.search(searchTerm);
        }
        
        LogMgr.log(LOG, LogLevel.PROCESSING1, "Number of messages to be processed in ", folder.getName(), ": ", messages.length);
        
        for(Message message : messages)
        {
            processMessage(message, uidFolder, messageIDs);
        }
        LogMgr.log(LOG, LogLevel.PROCESSING1, "readMessagesInFolder completed for ", folder.getName());
    }
    
    private static SearchTerm getSearchTerm ()
    {
        if(!SYNC_ALL_EXISTING_EMAILS)
        {
            SearchTerm  receivedFrom    =   new ReceivedDateTerm(ComparisonTerm.GE, getSyncingFromDate());
            SearchTerm  sentFrom        =   new SentDateTerm(ComparisonTerm.GE, getSyncingFromDate());

            return new AndTerm(receivedFrom, sentFrom);
        }
        return null;
    }
    
    public static void processMessage(Message message, UIDFolder uidFolder, final Set<String> messageIDs)
    {
        LogMgr.log(LOG, LogLevel.PROCESSING2 , "processMessage called");
        
        String  tmpFromAdress   =   null;
        String  tmpEmailText;
        String  tmpSubject;
        Date    tmpSentDate;
        String  tmpRecipient    =   null;
        String  tmpJobId        =   null;
        String  tmpMessageID    =   null;
        
        List<FileBinaryContent> tmpContents;

        final String fromAddress;
        final String recipient;
        final String subject;
        final Date  sentDate;
        final String emailText;
        final String jobIdentifier;
        final String messageID;
        final List<FileBinaryContent> contents;

        try
        {
            if(message.getFrom() != null && message.getFrom().length > 0)
            {
                tmpFromAdress   =   ((InternetAddress) message.getFrom()[0]).getAddress();
            }
            
            tmpSubject      =   message.getSubject();
            tmpSentDate     =   message.getReceivedDate();
            tmpMessageID    =   ((MimeMessage)message).getMessageID();
                    
            LogMgr.log(LOG, LogLevel.PROCESSING2 , "Mail Subject:" + tmpSubject, " Address:", tmpFromAdress, " MessageId:", tmpMessageID);
            LogMgr.log(LOG, LogLevel.PROCESSING2 , "Mail Details: Received ", message.getReceivedDate(), " Sent:", message.getSentDate());
            
            
            if(message.getAllRecipients() != null && message.getAllRecipients().length > 0)
            {
                tmpRecipient = ((InternetAddress) message.getAllRecipients()[0]).getAddress();
            }            
            
            if(StringUtils.subBlanks(tmpRecipient) != null)
            {
                tmpJobId    =   getJobIdentifierFromEmail(tmpRecipient);
            }
            
            tmpEmailText    =   EmailFetcher.getText(message, new ArrayList<>());
            tmpContents     =   getAttachments(message);
        }
        catch (MessagingException | IOException ex)
        {
            LogMgr.log(LOG, LogLevel.PROCESSING2 , "Batch Email Fetcher : Mail message composing error", ex);
            throw new NestedException(ex);
        }

        fromAddress         =   tmpFromAdress;
        subject             =   tmpSubject;
        sentDate            =   tmpSentDate;
        jobIdentifier       =   tmpJobId;
        recipient           =   tmpRecipient;
        contents            =   tmpContents;
        emailText           =   tmpEmailText;
        messageID           =   tmpMessageID;
        
        try
        {
            if(!messageIDs.contains(messageID))
            {
                TransactionServices.run(new TransactionTask()
                {
                    @Override
                    public void run(ObjectTransaction objTran) throws FieldException, StorageException
                    {
                        if (jobIdentifier != null && !jobIdentifier.isEmpty())
                        {
                            Job     job =   Job.getJobByID(objTran, Long.parseLong(jobIdentifier));

                            if (job != null)
                            {
                                EmailMessage    emailMessage    =   EmailMessage.createEmailMessage(objTran);
                                
                                emailMessage.setMessageId(messageID);
                                emailMessage.setEmailFrom(fromAddress);
                                emailMessage.setEmailTo(recipient);
                                emailMessage.setSubject(subject);
                                emailMessage.setDescription(emailText);
                                emailMessage.setReceivedDate(sentDate);
                                emailMessage.setJob(job);
                                
                                for (FileBinaryContent content : contents)
                                {
                                    Attachment  attachment  =   Attachment.createAttachment(objTran);
                                    
                                    attachment.setAttachmentFile(content);
                                    emailMessage.addToAttachments(attachment);
                                }
                            }
                        }
                    }
                });
                
                LogMgr.log(LOG, LogLevel.PROCESSING2 , "processMessage completed successfully");
                messageIDs.add(messageID);
            }
            else
            {
                LogMgr.log(LOG, LogLevel.PROCESSING2 , "Message already synced or doesnt have valid sent time ", sentDate);
            }
        }
        catch (Exception e)
        {
            LogMgr.log(LOG, LogLevel.PROCESSING2 , e, "Batch Email Fetcher : Error in creating the email message ");
        }
    }
    
    public static String getJobIdentifierFromEmail(String strReceipient)
    {
        int index = strReceipient.indexOf('@');
        
        return strReceipient.substring(0, index);
    }
    
    private static List<FileBinaryContent> getAttachments(Message message) throws IOException, MessagingException
    {
        List<FileBinaryContent> contents    =   new ArrayList<>();
        
        if(message.getContentType() != null && message.getContentType().contains("multipart"))
        {
            Multipart               multipart   =   (Multipart)message.getContent();
            byte[]                  bytes       =   null;

            for (int j = 0; j < multipart.getCount(); j++) 
            {
                BodyPart bodyPart = multipart.getBodyPart(j);

                if(Part.ATTACHMENT.equalsIgnoreCase(bodyPart.getDisposition())) 
                {
                    bytes   =   IOUtils.readInputStreamToBytes(bodyPart.getInputStream());

                    FileBinaryContent binaryContent = new FileBinaryContent(bodyPart.getContentType(), bytes, bodyPart.getFileName());
                    contents.add(binaryContent);
                }
            }
        }
        return contents;
    }
    
    private static Date getSyncingFromDate()
    {
        return DateDiff.before(new Date(), SYNC_EMAILS_SINCE_MINUTES, Calendar.MINUTE);
    }
    
    @Override
    public void init(ParticipantInitialisationContext pic) throws InitialisationException 
    {
        throw new UnsupportedOperationException("Not supported yet.");
    }

}
