package performa.orm;

import java.util.*;
import oneit.logging.LoggingArea;
import oneit.objstore.*;
import oneit.objstore.cloning.AssocCopyingRule;
import oneit.objstore.cloning.BusinessCopyHelper;
import oneit.objstore.cloning.CopyType;
import oneit.objstore.cloning.StandardAssocCopyingRule;
import oneit.objstore.rdbms.filters.*;
import oneit.objstore.utils.ObjstoreUtils;
import oneit.security.*;
import oneit.utils.*;
import oneit.utils.filter.CollectionFilter;
import oneit.utils.filter.Filter;
import oneit.utils.parsers.FieldException;
import performa.orm.types.*;
import performa.utils.Utils;


public class Job extends BaseJob
{
    private static final long serialVersionUID = 0L;
   
    public  static LoggingArea  LOG = LoggingArea.createLoggingArea("Job");

    
    // This constructor should not be called
    public Job ()
    {
        // Do not add any code to this, always put it in initialiseNewObject
    }

    
    @Override
    protected void postInitNewObj() throws BusinessException 
    {
        super.postInitNewObj();
        
        setRandomKey(RandomStringGen.getRandomStringGen().generateAlphaNum(4));
        setSecUser(SecUser.getTXUser(getTransaction()));
        setApplyBy(DateDiff.add(DateDiff.getToday(), Calendar.DATE, 30));
        
        for(CultureElement cultureElement : CultureElement.searchAll(getTransaction()))
        {
            CultureCriteria cultureCriteria =   CultureCriteria.createCultureCriteria(getTransaction());
            
            cultureCriteria.setCultureElement(cultureElement);
            addToCultureCriterias(cultureCriteria);
        }
    }
    
    
    @Override
    public void validate(ValidationContext context) 
    {
        super.validate(context);

        if (getIncludeAssessmentCriteria())
        {
           context.check(getAssessmentCriteriasCount() > 0 ,  this, MULTIPLEREFERENCE_AssessmentCriterias, "atleastOneRequirement");
        }
    }
    
    
    public Boolean jobDetailsCompleted()
    {
        return getJobTitle() != null && getJobDescription() != null;
    }
    
    
    public Boolean assessmentCompleted()
    {
        return Boolean.TRUE;
    }
    
    
    public Boolean cultureCompleted()
    {
        return Boolean.FALSE;
    }
    
    
    public Boolean jobMatchCompleted()
    {
        return Boolean.TRUE;
    }
    
    
    public Collection<AssessmentCriteria> getAssessmentCriteraByType(CriteriaType criteria)
    {
        Filter  filter  = AssessmentCriteria.SearchByAll().andCriteriaType(new EqualsFilter<>(criteria));   

        return  CollectionFilter.filter(getAssessmentCriteriasSet(), filter);
    }
    
    
    public int getAssessmentCriteraCountByType(CriteriaType criteria)
    {
        return getAssessmentCriteraByType(criteria).size();
    }
    
    
    //to get All Profile Assessment Questions
    public Collection<Question> getAllProfileAssessmentQuestions()
    {
        return pipelineJob().toLevel().toFactors().toFactor().toQuestions().toQuestion().uniqueVals();
    }
    
    
    //to get only left Questions
    public List<Question> getAllLeftQuestions()
    {
        Filter  filter  = Question.SearchByAll().andIsRightQuestion(new EqualsFilter<>(Boolean.FALSE));
        
        return  ObjstoreUtils.sort(CollectionFilter.filter(getAllProfileAssessmentQuestions(), filter), 
                                new ObjectTransform[] {Question.pipesQuestion().toSection().toObjectID(), Question.pipesQuestion().toObjectID()},  
                                new Comparator[]{CollectionUtils.DEFAULT_COMPARATOR, CollectionUtils.DEFAULT_COMPARATOR});
    }
  
    
    public Collection<CultureCriteria> getCultureCriteriasForClass(CultureClass cultureClass)
    {
        Filter  elementFilter   =   CultureElement.SearchByAll().andCultureClass(new EqualsFilter(cultureClass));
        
        Set<CultureElement> cultureElements =   pipelineJob().toCultureCriterias().toCultureElement(elementFilter).uniqueVals();
        
        Filter  criteriaFilter  =   CultureCriteria.SearchByAll().andCultureElement(new InFilter(cultureElements));
        
        return pipelineJob().toCultureCriterias(criteriaFilter).vals();
    }
    
    
    public Level[] getAllLevels()
    {     
        List<Level> levels  =   new ArrayList<>();
        
        Utils.getLevelsForComprehensive(getTransaction()).stream().forEach((tuple) -> {
            levels.add((Level)tuple.get0());
        });
        return levels.toArray(new Level[0]);
    }

    public int getNoOfCandidatesApplied()
    {
        return getNoOfCandidatesFor(ApplicationStatus.SUBMITTED);
    }
    
    public int getNoOfCandidatesShortlisted()
    {
        return getNoOfCandidatesFor(ApplicationStatus.SHORTLISTED);
    }
    
    public int getNoOfCandidatesFor(ApplicationStatus status)
    {
        Filter<JobApplication>  filter = JobApplication.SearchByAll().andApplicationStatus(new EqualsFilter<>(status));
        
        return pipelineJob().toJobApplications(filter).toCandidate().vals().size();
    }
    
    
    public int  getNoOfDaystoClosed()
    {
        int dateDiff    =   0;
        
        if(getJobStatus()==JobStatus.OPEN && getApplyBy()!=null)
        {
            dateDiff    =   DateDiff.getDateDiff(Calendar.DATE, DateDiff.getToday(), getApplyBy());
        }
        
        return dateDiff>0 ? dateDiff : 0;
    }
    
    
    public String getDaysTillClose()
    {
        StringBuilder   sb  =   new StringBuilder();

        if(getApplyBy()!=null)
        {
            int dateDiff    =   getNoOfDaystoClosed();

            sb.append(dateDiff);

            if(dateDiff==1)
            {
                sb.append(" day till close");
            }
            else
            {
                sb.append(" days till close");
            }
        }
        return sb.toString();
    }
    
    
    public String getClosingInText()
    {
        StringBuilder   sb  =   new StringBuilder("Closing in ");

        if(getApplyBy()!=null)
        {
            int dateDiff    = getNoOfDaystoClosed();

            sb.append(dateDiff);

            if(dateDiff==1)
            {
                sb.append(" day");
            }
            else
            {
                sb.append(" days");
            }
        }
        return sb.toString();
    }
    
    
    public String getJobStatusText()
    {
        StringBuilder   sb  =   new StringBuilder("THIS JOB IS ");
        
        sb.append(getJobStatus().getDescription());
        
        return sb.toString();
    }
    
    
    public AssessmentCriteriaTemplate[] getAssessmentTemplates()
    {
        return AssessmentCriteriaTemplate.SearchByAll()
                                        .andUser(new EqualsFilter<>(SecUser.getTXUser(getTransaction())))
                                        .search(getTransaction());
    }
    
    
    public CultureCriteriaTemplate[] getCultureTemplates()
    {
        return CultureCriteriaTemplate.SearchByAll()
                                        .andUser(new EqualsFilter<>(SecUser.getTXUser(getTransaction())))
                                        .search(getTransaction());
    }

    
    @Override
    public String getToString() 
    {
        return getJobTitle();
    }
    
    
    public Job cloneJob(Job newJob) throws FieldException
    {
        return (Job) BusinessCopyHelper.copyBusinessObject(this, newJob, null, new HashMap(), new AssocCopyingRule()
        {
            @Override
            protected BaseBusinessClass getVersionToReferTo (BaseBusinessClass from, String assocName, BaseBusinessClass to, Map newObjectRefs, CopyType copyType)
            {
                if (newObjectRefs.containsKey (to.getObjectID ()))
                {
                    return (BaseBusinessClass)newObjectRefs.get (to.getObjectID ());
                }
                else if (to instanceof AssessmentCriteria || to instanceof CultureCriteria )
                {
                    /* Create a copy of 'to' */
                    return BusinessCopyHelper.copyBusinessObject (to, copyType, newObjectRefs, this);
                }
                else
                {
                    /* Do not create a copy of 'to' */
                    return to;
                }
            }
        });
    }
}