package performa.utils;

import java.awt.Color;
import java.util.*;
import javax.activation.DataSource;
import javax.servlet.http.HttpServletRequest;
import oneit.appservices.config.ConfigMgr;
import oneit.business.content.Article;
import oneit.email.*;
import oneit.logging.*;
import oneit.net.LoopbackHTTP;
import oneit.objstore.*;
import oneit.objstore.rdbms.filters.*;
import oneit.objstore.utils.ObjstoreUtils;
import oneit.security.*;
import oneit.security.jsp.*;
import oneit.security.oauth.form.*;
import oneit.servlets.forms.*;
import oneit.servlets.orm.DataMap;
import oneit.servlets.process.ORMProcessState;
import oneit.servlets.security.SessionSecUserDecorator;
import oneit.utils.*;
import oneit.utils.filter.*;
import oneit.utils.parsers.FieldException;
import oneit.utils.transform.MapTransform;
import oneit.utils.transform.param.*;
import performa.chart.RingChart;
import performa.form.LinkedInOAuthLoginFP;
import performa.orm.*;
import performa.orm.types.*;
import performa.search.SearchApplicant;

/**
 *
 * @author Harsh
 */
public class Utils
{
    public static final String  ROLE_APPLICANT                  =   "TL_Applicant";
    public static final String  ROLE_CLIENT                     =   "TL_Client";
    public static final String  ROLE_COMPANY_ADMIN              =   "TL_CompanyAdmin";
    public static final String  ROLE_COMPANY_USER               =   "TL_CompanyUser";
    public static final String  PRIV_ACCESS_ADMIN_PORTAL        =   "TL_AccessAdminPortal";
    public static final String  PRIV_ACCESS_APPLICANT_PORTAL    =   "TL_AccessApplicantPortal";
    public static final String  PRIV_ACCESS_COMPANY_ADMIN       =   "TL_AccessCompanyAdmin";
    public static final String  PRIV_ACCESS_COMPANY             =   "TL_AccessCompany";
    public static final String  ADVT_CONFIG_EMAIL_EXTENSION     =   ConfigMgr.getKeyfileString ("advt.config.email.extension", "matchdmail.com");
    public static final String  APPLICANT_VERIFY_LINK           =   "/ApplicantPortal-Applicant_Account_Verification.htm";
    public static final String  APPLICANT_LINK                  =   "/ApplicantPortal-ApplyJob.htm";

    public static Role  getRole(String role, ObjectTransaction transaction)
    {
        return Role.SearchByNAME().byName(role).search(transaction);
    }
    
    public static Privilege  getPrivilege(String priv, ObjectTransaction transaction)
    {
        return Privilege.searchNAME(transaction, priv);
    }
    
    public static boolean checkAdminPortalAccess(SecUser secUser)
    {
        return (secUser != null && secUser.hasPrivilege(PRIV_ACCESS_ADMIN_PORTAL));
    }
    
    public static boolean checkApplicantPortalAccess(SecUser secUser)
    {
        return (secUser != null && secUser.hasPrivilege(PRIV_ACCESS_APPLICANT_PORTAL));
    }
    
    public static Candidate getCandidateUser (ObjectTransaction objTran)
    {
        return  SecUser.getTXUser(objTran).getExtension(Candidate.REFERENCE_Candidate);
    }
    
    public static Importance[] getImportancesForAssessment()
    {
        List<Importance>    importances =   new ArrayList<>();
        
        for(Importance importance : Importance.getImportanceArray())
        {
            if(importance.getConsiderForAssessment())
            {
                importances.add(importance);
            }
        }
        
        Collections.sort(importances, (a, b) -> b.getWeightingScore().compareTo(a.getWeightingScore()));
       
        return importances.toArray(new Importance[0]);
    }
    
    public static Importance[] getImportancesForCulture()
    {
        List<Importance>    importances =   new ArrayList<>();
        
        for(Importance importance : Importance.getImportanceArray())
        {
            if(importance.getConsiderForAssessment())
            {
                importances.add(importance);
            }
        }
//        Collections.sort(importances, (a, b) -> b.getWeightingScore().compareTo(a.getWeightingScore()));
       
        return importances.toArray(new Importance[0]);
    }
    
    //to sort jobs list
    public static List<Job> getJobsSorted(List<Job> jobs, JobSortOption jobSortOption, JobStatus jobStatus)
    {
        ObjectTransform transform   =   Job.pipesJob().toObjectCreated();
        Comparator      comparator  =   CollectionUtils.DEFAULT_COMPARATOR_NULLS_FIRST;
        
        if(jobSortOption == JobSortOption.OLDEST || jobSortOption == JobSortOption.NEWEST)
        {
            if(jobStatus == null)
            {
                transform   =   Job.pipesJob().toObjectCreated();
            }
            else if(jobStatus == JobStatus.OPEN)
            {
                transform   =   Job.pipesJob().toOpenDate();
            }
            else
            {
                transform   =   Job.pipesJob().toLastStatusChangeDate();
            }
            
            comparator  =   jobSortOption == JobSortOption.OLDEST  ? 
                                CollectionUtils.DEFAULT_COMPARATOR_NULLS_FIRST : 
                                CollectionUtils.reverse(CollectionUtils.DEFAULT_COMPARATOR_NULLS_FIRST);
        }
        else if(jobSortOption == JobSortOption.CLOSING_SOON)
        {
            transform   =   Job.pipesJob().toApplyBy();
            comparator  =   CollectionUtils.DEFAULT_COMPARATOR_NULLS_FIRST;
        }
        else if(jobSortOption == JobSortOption.ALPHA_A_Z)
        {
            transform   =   Job.pipesJob().toJobTitle();
            comparator  =   CollectionUtils.CASE_INSENSITIVE_COMPARATOR_NULLS_FIRST;
        }
        else if(jobSortOption == JobSortOption.ALPHA_Z_A)
        {
            transform   =   Job.pipesJob().toJobTitle();
            comparator  =   CollectionUtils.reverse(CollectionUtils.CASE_INSENSITIVE_COMPARATOR_NULLS_FIRST);
        }
        
        return ObjstoreUtils.sort( jobs, 
                        new ObjectTransform[]{transform},
                        new Comparator[]{comparator});
    }
    

    public static List<Client> getClientsSorted(Client[] clients, ClientSortOption clientSortOption)
    {
        ObjectTransform transform   =   Client.pipesClient().toObjectCreated();
        Comparator      comparator  =   CollectionUtils.DEFAULT_COMPARATOR_NULLS_FIRST;
        
        if(clientSortOption == ClientSortOption.ALPHA_A_Z)
        {
            transform   =   Client.pipesClient().toClientName();
            comparator  =   CollectionUtils.CASE_INSENSITIVE_COMPARATOR_NULLS_FIRST;
        }
        else if(clientSortOption == ClientSortOption.ALPHA_Z_A)
        {
            transform   =   Client.pipesClient().toClientName();
            comparator  =   CollectionUtils.reverse(CollectionUtils.CASE_INSENSITIVE_COMPARATOR_NULLS_FIRST);
        }
        else if(clientSortOption == ClientSortOption.OPEN_JOBS)
        {
            transform   =   new ClientJobCountTransform(JobStatus.OPEN);
            comparator  =   CollectionUtils.reverse(CollectionUtils.DEFAULT_COMPARATOR_NULLS_FIRST);
        }
        else if(clientSortOption == ClientSortOption.FILLED_JOBS)
        {
            transform   =   new ClientJobCountTransform(JobStatus.FILLED);
            comparator  =   CollectionUtils.reverse(CollectionUtils.DEFAULT_COMPARATOR_NULLS_FIRST);
        }
        
        return ObjstoreUtils.sort( Arrays.asList(clients), new ObjectTransform[]{transform}, new Comparator[]{comparator});
    }
    
    
    public static Collection<CompanyUser> getUsersSorted(Collection<CompanyUser> users, UserSortOption userSortOption)
    {
        Comparator  comparator  =   (userSortOption == UserSortOption.ALPHA_Z_A) ? CollectionUtils.reverse(CollectionUtils.DEFAULT_COMPARATOR_NULLS_FIRST) : CollectionUtils.DEFAULT_COMPARATOR_NULLS_FIRST;
        
        return ObjstoreUtils.sort(users, new ObjectTransform[]{CompanyUserTransform.INSTANCE}, new Comparator[]{comparator});
    }
    
    
    public static Collection<HiringTeam> getHiringTeamsSorted(Collection<HiringTeam> hiringTeams, UserSortOption userSortOption)
    {
        Comparator  comparator  =   (userSortOption == UserSortOption.ALPHA_Z_A) ? CollectionUtils.reverse(CollectionUtils.DEFAULT_COMPARATOR_NULLS_FIRST) : CollectionUtils.DEFAULT_COMPARATOR_NULLS_FIRST;
        
        return ObjstoreUtils.sort(hiringTeams, new ObjectTransform[]{HiringTeam.pipesHiringTeam().toHiringTeamName()}, new Comparator[]{comparator});
    }
    
    
    public static class ClientJobCountTransform implements ObjectTransform<Client, Integer>
    {
        JobStatus   jobStatus;

        public ClientJobCountTransform(JobStatus jobStatus)
        {
            this.jobStatus = jobStatus;
        }
        
        @Override
        public Integer transform(Client client)
        {
            return client.getJobCountByStatus(jobStatus);
        }
    }
    
    public static int getClosingSoonJobCount(Job[] jobs)
    {
        Filter<Job> filter          =   Job.SearchByAll()
                                            .andApplyBy(new LessThanFilter<>(DateDiff.add(DateDiff.getToday(), Calendar.DATE, 5)))
                                            .andApplyBy(new GreaterThanEqualFilter<>(DateDiff.getToday()));
        
        Object[]    closingSoonJobs = CollectionFilter.filterArray(jobs, filter);
        
        return closingSoonJobs.length;
    }
    
    //to sort application list
    public static List<JobApplication> getApplicationsSorted(List<JobApplication> applications, AppSortOption appSortOption)
    {
        ObjectTransform transform   =   JobApplication.pipesJobApplication().toObjectCreated();
        Comparator      comparator  =   CollectionUtils.DEFAULT_COMPARATOR_NULLS_FIRST;
        
        if(appSortOption==AppSortOption.OLDEST)
        {
            transform   =   JobApplication.pipesJobApplication().toSubmittedDate();
            comparator  =   CollectionUtils.DEFAULT_COMPARATOR_NULLS_FIRST;
        }
        else if(appSortOption==AppSortOption.NEWEST)
        {
            transform   =   JobApplication.pipesJobApplication().toSubmittedDate();
            comparator  =   CollectionUtils.reverse(CollectionUtils.DEFAULT_COMPARATOR_NULLS_FIRST);
        }
        else if(appSortOption==AppSortOption.RANK) 
        {
            transform   =   JobApplication.pipesJobApplication().toOverallRank();
            comparator  =   CollectionUtils.DEFAULT_COMPARATOR_NULLS_FIRST;
        }
        else if(appSortOption==AppSortOption.ALPHA_A_Z)
        {
            transform   =   JobApplication.pipesJobApplication().toCandidate().toUser().toFirstName();
            comparator  =   CollectionUtils.CASE_INSENSITIVE_COMPARATOR_NULLS_FIRST;
        }
        else if(appSortOption==AppSortOption.ALPHA_Z_A)
        {
            transform   =   JobApplication.pipesJobApplication().toCandidate().toUser().toFirstName();
            comparator  =   CollectionUtils.reverse(CollectionUtils.CASE_INSENSITIVE_COMPARATOR_NULLS_FIRST);
        }
        
        return ObjstoreUtils.sort(applications, 
                                new ObjectTransform[]{transform},
                                new Comparator[]{comparator});
    }
    
    
    public static ObjectTransform createCompoundTransform(Map defaultTransMap, BaseBusinessClass... bbcs)
    {
        PrefixCompoundTransform prefixTransform =   new PrefixCompoundTransform();
        
        if(defaultTransMap != null)
        {
            prefixTransform.setDefault(new MapTransform(defaultTransMap));
        }
        
        for(BaseBusinessClass bbc : bbcs)
        {
            if(bbc != null)
            {
                prefixTransform.add(bbc.getClass().getSimpleName(), new ORMTransform(bbc));
            }
        }
        
        return new StringUtils.NullToBlankPostTransform(new ErrorTransform(prefixTransform, ""));
    }
    
    
    public static void sendMail(ConfigurableEmailer emailer, ObjectTransform finalTransform, String[] emails, DataSource[] attachments, BaseBusinessClass bo) throws ConfigurableEmailerException
    {
        LogMgr.log(LoggingArea.ALL, LogLevel.PROCESSING1, "Sending Mail from Utils class to" + Arrays.toString(emails));
        
        emailer.sendTransactionalEmail(bo.getTransaction(), 
                new Date(), 
                finalTransform, 
                emails, 
                null, 
                null, 
                null, 
                null, 
                null, 
                null, 
                attachments, 
                bo.getClass().getName(), 
                bo.getObjectID());
        
        LogMgr.log(LoggingArea.ALL, LogLevel.PROCESSING1, "Mail sent from Utils class to " + Arrays.toString(emails));
    }
    
    
    public static String getPwdKeyOfSecUser(HttpServletRequest request, SecUser user, boolean mandatoryPwd)
    {
        if(user != null)
        {
            DataMap dm  =   DataMap.getDataMap(request, true);
            
            return dm.storeORMHandler(new PasswordDIHandler(user, "md5:" + SecUser.FIELD_Password, mandatoryPwd), user, "md5:" + SecUser.FIELD_Password);
        }
        
        return "";
    }
    
        
    public static SuccessfulResult processSuccessfulLogin(ORMProcessState process, SubmissionDetails submission, Map params, SecUser user) throws BusinessException
    {
        HttpServletRequest  request =   submission.getRequest();
        
        request.getSession().setAttribute (SecUser.SEC_USER_ID, user);
        request.getSession().setAttribute (SessionSecUserDecorator.REFRESH_SECURITY, Boolean.TRUE);
        
        process.completeAndRestart();
        
        return new RedirectResult((String) request.getAttribute("nextPage"), null);
    }
    
    
    public static Client[] getClientsByHiringTeam(ObjectTransaction transaction)
    {
        SecUser     secUser     =   SecUser.getTXUser(transaction);
        CompanyUser companyUser =   secUser.getExtension(CompanyUser.REFERENCE_CompanyUser);
        
        return companyUser.getSelectedTeam().getClientsSet().toArray(new Client[0]);
    }
    
    
    public static Boolean emailExists(ObjectTransaction objTran, String email)
    {
        if(email!= null)
        {
            return SecUser.searchNAME(objTran, email.toLowerCase()) != null;
        }
        
        return Boolean.FALSE;
    }
    
    
    public static Boolean isCompanyUserEmailFound(ObjectTransaction objTran, String email)
    {
        if(email!=null)
        {
            SecUser user    = SecUser.searchNAME(objTran, email.toLowerCase());
            
            if(user!=null && user.getExtension(CompanyUser.REFERENCE_CompanyUser)!=null)
            {
                return Boolean.TRUE;
            }
        }
        
        return Boolean.FALSE;
    }
    
    public static Boolean userFoundInAnotherCompany(ObjectTransaction objTran, String email, Company company)
    {
        if(email != null)
        {
            SecUser user    =   SecUser.searchNAME(objTran, email.toLowerCase());
            
            if(user != null && user.getExtension(CompanyUser.REFERENCE_CompanyUser) != null)
            {
                CompanyUser companyUser =   user.getExtension(CompanyUser.REFERENCE_CompanyUser);

                return !CollectionUtils.equals(companyUser.getCompany(), company);
            }
        }
        
        return Boolean.FALSE;
    }
    
    public static Boolean isCompanyUserEmailFound(ObjectTransaction objTran, String email, CompanyUser currentUser)
    {
        if(email!=null)
        {
            SecUser user    = SecUser.searchNAME(objTran, email.toLowerCase());

            if(user!=null)
            {
                CompanyUser comUser = user.getExtension(CompanyUser.REFERENCE_CompanyUser); 
                
                if(comUser!=null && !CollectionUtils.equals(comUser, currentUser))
                {
                    return Boolean.TRUE;
                }
            }
        }

        return Boolean.FALSE;
    }
    
    
    public static Tuple.T2<SecUser, Boolean> getSecUserForCompanyIfAvailable(CompanyUser companyUser)
    {
        SecUser     secUser             =   companyUser.getUser();
        SecUser     availableSecUser    =   SecUser.searchNAME(companyUser.getTransaction(), secUser.getEmail());

        if(availableSecUser!=null)
        {
            if(availableSecUser.getExtension(CompanyUser.REFERENCE_CompanyUser)==null)
            {

                availableSecUser.addToExtensions(companyUser);

                //remove new sec user obj
                secUser.removeFromExtensions(companyUser);
                secUser.delete();

                return new Tuple.T2<SecUser, Boolean>(availableSecUser, Boolean.TRUE);
            }
        }
        
        return new Tuple.T2<SecUser, Boolean>(secUser, Boolean.FALSE);
    }
    
    
    public static void sendVerificationMail(Job job, CompanyUser companyUser, HttpServletRequest request, ConfigurableArticleTemplateEmailer  emailer, String callingClass) throws BusinessException
    {
        if(!CollectionUtils.equals(companyUser.getIsAccountVerified(), Boolean.TRUE))
        {
            try
            {
                LogMgr.log(CompanyUser.LOG, LogLevel.PROCESSING1, "Sending verification mail from  " + callingClass + " to :: ", companyUser);

                Article         verificationArticle =   WebUtils.getArticleByShortCut(companyUser.getTransaction(), WebUtils.COMPANY_ACCOUNT_VERIFICATION);
                RandomStringGen random              =   new RandomStringGen();

                //set verification key and send mail time
                companyUser.setVerificationKey(random.generateAlphaNum(6));
                companyUser.setVerificationMailSendDate(new Date());

                String          link                =   LoopbackHTTP.getRemoteAccessURL(request) 
                                                            + verificationArticle.getLink(request, CollectionUtils.EMPTY_MAP, "/") 
                                                            + "?id=" + companyUser.getID()
                                                            + "&key=" + companyUser.getVerificationKey();
                Map             defaultParams       =   CollectionUtils.mapEntry("link", link).toMap(); 
                ObjectTransform transform           =   Utils.createCompoundTransform(defaultParams, companyUser.getUser(), job);

                Utils.sendMail(emailer, transform, new String[]{companyUser.getEmailAddressFromUser()}, null, companyUser);

                LogMgr.log(CompanyUser.LOG, LogLevel.PROCESSING1, "Sent verification mail successfully from " + callingClass + " to :: ", companyUser);
            }
            catch (ConfigurableEmailerException ex)
            {
                LogMgr.log(CompanyUser.LOG, LogLevel.SYSTEMERROR1, ex, "Error occured while sending mail for user :: " + companyUser);

                throw new BusinessException("We are unable to send mail. Please try again or contact Matchd for more details.");
            }
        }
        else
        {
            LogMgr.log(CompanyUser.LOG, LogLevel.PROCESSING1, "Call from " + callingClass + ". Account is already verified  for user :: ", companyUser);
        }
    }
    
    
    public static void sendEmailChangedMail(CompanyUser companyUser, HttpServletRequest request, ConfigurableArticleTemplateEmailer  emailer, String callingClass) throws BusinessException
    {
        if(companyUser.getIsEmailChanged()==Boolean.TRUE)
        {
            try
            {
                LogMgr.log(CompanyUser.LOG, LogLevel.PROCESSING1, "Sending email changed mail from  " + callingClass + " to :: ", companyUser);

                RandomStringGen random              =   new RandomStringGen();

                //set verification key and send mail time
                companyUser.setVerificationKey(random.generateAlphaNum(6));
                companyUser.setVerificationMailSendDate(new Date());

                String          link                =   LoopbackHTTP.getRemoteAccessURL(request) 
                                                            + "/extensions/adminportal/email_verified.jsp"
                                                            + "?id=" + companyUser.getID()
                                                            + "&key=" + companyUser.getVerificationKey();
                Map             defaultParams       =   CollectionUtils.mapEntry("link", link).toMap(); 
                ObjectTransform transform           =   Utils.createCompoundTransform(defaultParams, companyUser.getUser());

                Utils.sendMail(emailer, transform, new String[]{companyUser.getUser().getEmail()}, null, companyUser);

                LogMgr.log(CompanyUser.LOG, LogLevel.PROCESSING1, "Sent email changed mail successfully from " + callingClass + " to :: ", companyUser);
            }
            catch (ConfigurableEmailerException ex)
            {
                LogMgr.log(CompanyUser.LOG, LogLevel.SYSTEMERROR1, ex, "Error occured while sending mail for user :: " + companyUser);

                throw new BusinessException("We are unable to send mail. Please try again or contact Matchd for more details.");
            }
        }
        else
        {
            LogMgr.log(CompanyUser.LOG, LogLevel.PROCESSING1, "Call from " + callingClass + ". Account is already verified  for user :: ", companyUser);
        }
    }
    
    public static boolean facebookAvailable()
    {
        return  ConfigMgr.getKeyfileString(FacebookOAuthLoginFP.FACEBOOK_APPID_ATTRIB_NAME) != null 
                && ConfigMgr.getKeyfileString(FacebookOAuthLoginFP.FACEBOOK_APP_SECRET_ATTR_NAME) != null;
    }   
    
    
    public static boolean linkedInAvailable()
    {
        return  ConfigMgr.getKeyfileString(LinkedInOAuthLoginFP.LINKEDIN_CLIENTID_ATTR_NAME) != null 
                && ConfigMgr.getKeyfileString(LinkedInOAuthLoginFP.LINKEDIN_CLIENTSECRET_ATTR_NAME) != null;
    }   
    
    
    public static boolean googleAvailable()
    {
        return  ConfigMgr.getKeyfileString(GoogleOAuthLoginFP.GOOGLE_CLIENTID_ATTR_NAME) != null 
                && ConfigMgr.getKeyfileString(GoogleOAuthLoginFP.GOOGLE_CLIENTSECRET_ATTR_NAME) != null;
    }
    
    
    //to get Job status list sorted for client
    public static Collection<JobStatus>  getJobStatusesForClient()
    {
         return CollectionUtils.sort(Arrays.asList(JobStatus.getJobStatusArray()), 
                            new Comparator<JobStatus> () {
                                @Override
                                public int compare(JobStatus o1, JobStatus o2)
                                {
                                    return o1.getClientSortOrder().compareTo(o2.getClientSortOrder());
                                }
                            });
    }
    
    private static class CompanyUserTransform implements ObjectTransform<CompanyUser, String>
    {
        static  CompanyUserTransform    INSTANCE    =   new CompanyUserTransform();
        
        @Override
        public String transform(CompanyUser companyUser)
        {
            return SecUserToNameTransform.INSTANCE.transform(companyUser.getUser());
        }
    }
    
    
    public static PaymentPlan[] getPaymentPlansForJobs(ObjectTransaction objTran)
    {
        return PaymentPlan.SearchByAll()
                        .andActiveJobCount(new IsNotNullFilter<>())
                        .search(objTran);
    }
    
    
    public static PaymentPlan getMaximumPaymentPlan(ObjectTransaction objTran)
    {
        return PaymentPlan.SearchByMax().search(objTran);
    }
    
    
    public static HiringTeam[] getHiringTeamsByUser(CompanyUser companyUser)
    {
        return companyUser.pipelineCompanyUser().toHiringTeams().toHiringTeam().uniqueVals().toArray(new HiringTeam[0]);
    }
    
    public static HiringTeam[] getOtherHiringTeams(HiringTeam hiringTeam, CompanyUser comUser, CompanyUserHiringTeamLink userLink)
    {
        Filter<HiringTeam>  filter  =   HiringTeam.SearchByAll().andObjectID(new NotEqualsFilter<>(hiringTeam.getID().longID()));

        return comUser.pipelineCompanyUser().toHiringTeams(CompanyUserHiringTeamLink.SearchByAll().andRole(new EqualsFilter<>(userLink.getRole())))
                .toHiringTeam(filter).uniqueVals().toArray(new HiringTeam[0]);
    }
    
    public static HiringTeam[] getHiringTeamsWithOwnBilling(HiringTeam hiringTeam)
    {
        Filter<HiringTeam>  filter  =   HiringTeam.SearchByBillingTeams()
                                                    .andManageOwnBilling(new EqualsFilter<>(Boolean.TRUE))
                                                    .andCardID(new IsNotNullFilter<>())
                                                    .andObjectID(new NotEqualsFilter<>(hiringTeam.getID().longID()));

        return hiringTeam.pipelineHiringTeam().toCompany().toHiringTeams(filter).uniqueVals().toArray(new HiringTeam[0]);
    }
    
    public static void setupJobFromTemplate(Job job, AssessmentCriteriaTemplate jobTemplate, ObjectTransaction objTran) throws FieldException, StorageException, BusinessException 
    {
        for(AssessmentCriteria  criteria : job.getAssessmentCriteriasSet())
        {
            criteria.delete();
        }
        
        for(WorkFlow  workflow : job.getWorkFlowsSet())
        {
            workflow.delete();
        }
        
        job.setJobTitle(jobTemplate.getJobTitle());
        job.setReferenceNumber(jobTemplate.getReferenceNumber());
        job.setJobDescription(jobTemplate.getJobDescription());
        job.setJobOwner(jobTemplate.getJobOwner());
        job.setClient(jobTemplate.getClient());
        
        job.setGoogleAddressText(jobTemplate.getGoogleAddressText());
        job.setExpectedCandidateRadius(jobTemplate.getExpectedCandidateRadius());
        job.setRemote(jobTemplate.getRemote());
        
        job.setJobType(jobTemplate.getJobType());
        job.setIndustry(jobTemplate.getIndustry());
        job.setILOJobTitle(jobTemplate.getILOJobTitle());
        job.setOccupation(jobTemplate.getOccupation());
        job.setRequireCV(jobTemplate.getRequireCV());
        
        job.setAssessmentType(jobTemplate.getAssessmentType());
        job.setLevel(jobTemplate.getLevel());
        
        for(AssessmentCriteria criteria : jobTemplate.getAssessmentCriteriasSet())
        {
            AssessmentCriteria criteriaCopy = AssessmentCriteria.createAssessmentCriteria(objTran);

            criteriaCopy.copyAttributesFrom(criteria);

            job.addToAssessmentCriterias(criteriaCopy);
        }
        
        for (WorkFlow workflow : jobTemplate.getWorkFlowsSet())
        {
            WorkFlow    workflowCopy    =    WorkFlow.createWorkFlow(objTran);

            workflowCopy.copyAttributesFrom(workflow);

            job.addToWorkFlows(workflowCopy);
        }
    }
    
    public static ApplicationStatus getApplicationStatus(int sortOrder)
    {
        for(ApplicationStatus status : ApplicationStatus.getApplicationStatusArray())
        {
            if(status.getSortOrder() == sortOrder)
            {
                return status;
            }
        }
        return null;
    }
    
    public static int getDayOfMonth(Date date)
    {
        Calendar cal = new GregorianCalendar ();

        cal.setTime(date);

        return cal.get(Calendar.DAY_OF_MONTH);
    }
    
    public static String getDayNumberSuffix(int day) 
    {
        if (day >= 11 && day <= 13) 
        {
            return "th";
        }
        
        switch (day % 10) 
        {
            case 1:
                return "st";
            case 2:
                return "nd";
            case 3:
                return "rd";
            default:
                return "th";
        }
    }
    
    public static Color getColor(ColorCode colorCode)
    {
        return colorCode == ColorCode.RED ? RingChart.RED : (colorCode == ColorCode.AMBER ? RingChart.AMBER : RingChart.GREEN);
    }
    
    public static String getCurrencyFormat(HiringTeam hiringTeam)
    {
        CurrencyType currency = hiringTeam.getCountry().getCurrency();
        
        return  currency == CurrencyType.GBP ? "CurrencyGBPOnly" : 
                currency == CurrencyType.EUR ? "CurrencyEuroOnly" :
                "CurrencyDollarsOnly";
    }
    
    
    public static List<RoleType> getAvailableRoles(CompanyUserHiringTeamLink userLink)
    {
        RoleType    roleType    =   userLink.getRole();
        
        if(roleType == RoleType.OWNER)
        {
            return Arrays.asList(new RoleType[]{RoleType.STANDARD, RoleType.ADMIN, RoleType.OWNER});
        }
        else if(roleType == RoleType.ADMIN)
        {
            return Arrays.asList(new RoleType[]{RoleType.STANDARD, RoleType.ADMIN});
        }
        
        return Arrays.asList(new RoleType[]{RoleType.STANDARD});
    }
    
    public static boolean isAssumedUser (HttpServletRequest request)
    {
        return request.getSession ().getAttribute (oneit.security.jsp.AssumeUserFP.UNASSUME_SEC_USER_ID) != null;
    }
    
    public static boolean isValidContentType(String contentType)
    {
        return  contentType.contains("msword") || contentType.contains("doc") || contentType.contains("docx") || contentType.contains("opendocument.text") 
                                || contentType.contains("pdf") || contentType.contains("vnd.openxmlformats-officedocument.wordprocessingml.document") || contentType.contains("text/plain");
    }
    
    public static String getApplicantSortingData(SearchApplicant searchApplicant)
    {
        StringBuilder sb = new StringBuilder();
        
        sb.append("{");
        sb.append("value: [");
        
        if(searchApplicant.getFilter() != null)
        {
            for(AppFilter filter : searchApplicant.getFilter())
            {
                if((filter == AppFilter.REQ || (filter.getParentNode() != null && filter.getParentNode() == AppFilter.REQ)) && searchApplicant.getJob().showAssessmentCriteriaSection())
                {
                    sb.append("'").append(filter.getName()).append("' ,");
                    continue;
                }
                
                DiversityQuestion question = searchApplicant.getJob().getQuestionByFilter(filter.getIsRoot() ? filter : filter.getParentNode());
                
                if(question != null)
                {
                    sb.append("'").append(filter.getName()).append("' ,");
                }
            }
        }
        sb.append("],");
        
        sb.append("options: [");
        
        for(AppFilter appFilter : AppFilter.getAppFilterArray())
        {
            DiversityQuestion question = searchApplicant.getJob().getQuestionByFilter(appFilter);
            
            if((!appFilter.getIsRoot() || question == null) && (appFilter != AppFilter.REQ || (appFilter == AppFilter.REQ && !searchApplicant.getJob().showAssessmentCriteriaSection())))
            {
                continue;
            }
            
            sb.append("{ id: '").append(appFilter.getName()).append("',");
            sb.append("label: '").append(appFilter.getDescription()).append("',");
            sb.append("children: [");
            
            for(AppFilter child : appFilter.getChildNodes())
            {
                sb.append("{id:'").append(child.getName()).append("', label:'").append(child.getDescription()).append("'},");
            }
            sb.append("]").append("},");
        }

        sb.append("]").append("}");
        
        return sb.toString();
    }
}